<?php

// Include Guzzle HTTP client classes for API communication
use GuzzleHttp\Client;
use GuzzleHttp\Exception\RequestException;

// Smart Plugin Generate Article

add_action('wp_ajax_smart_plugin_generate', 'wp_ajax_smart_plugin_generate_callback');

function wp_ajax_smart_plugin_generate_callback() {

    if (empty($_POST['api_key'])) {

        wp_send_json_error('<p>API key is missing.</p>');
    }

    $title         = sanitize_text_field($_POST['title']);
    $prompt        = sanitize_text_field($_POST['prompt']);
    $language      = sanitize_text_field($_POST['language']);
    $post_type     = sanitize_text_field($_POST['post_type']);
    $categories    = $_POST['categories'] ?? [];
    $model         = sanitize_text_field($_POST['model']);
    $article_image_id = sanitize_text_field($_POST['article_image_id']);
    $generateImage = sanitize_text_field($_POST['generateImage']);
    $api_key       = sanitize_text_field($_POST['api_key']);
    $site_url      = esc_url_raw($_POST['site_url']);

    $schedule_list = '';

    if (isset($_POST['schedule_list'])) {

        $schedule_list = intval($_POST['schedule_list']);

    }

    if( !empty($article_image_id) ) {

        $generateImage = false;
    }

    $client = new Client();

    try {

        $response = $client->post('https://smart-plugin.com/wp-json/generate-article/v1/receive-data/', [
            'headers' => [
                'Content-Type' => 'application/json',
            ],
            'json' => [
                'title'         => $title,
                'prompt'        => $prompt,
                'language'      => $language,
                'model'         => $model,
                'generateImage' => $generateImage,
                'api_key'       => $api_key,
                'site_url'      => $site_url,
            ],
            'timeout' => 60,
        ]);

        $code = $response->getStatusCode();

        $body = json_decode($response->getBody()->getContents(), true);

        if ($code !== 200) {

            wp_send_json_error('<p>Server error: HTTP ' . $code . '</p>');
        }

        if (json_last_error() !== JSON_ERROR_NONE || !is_array($body)) {

            wp_send_json_error('<p>Invalid JSON response from server.</p>');
        }

        if (!empty($body['status']) && $body['status'] === 'success') {

            $article_data = $body['data'];

            $post_title   = sanitize_text_field($article_data['title']);
            $post_content = wp_kses_post($article_data['content']);
            $post_tags    = $article_data['tags'];
            $post_categories    = $article_data['categories'];
            $post_image   = $article_data['generated_image'];
            $post_alt     = sanitize_text_field($article_data['image_alt_text']);
            $post_meta_title = sanitize_text_field($article_data['meta_title']);
            $post_meta_description = sanitize_text_field($article_data['meta_description']);

            $post_id = wp_insert_post([
                'post_title'   => $post_title,
                'post_content' => $post_content,
                'post_status'  => 'publish',
                'post_type'    => $post_type,
            ]);

            if (is_wp_error($post_id)) {
                
                wp_send_json_error('<p>Failed to create post: ' . esc_html($post_id->get_error_message()) . '</p>');
            }

            if( !empty($categories) ) {

                foreach ($categories as $taxonomy_key => $categories_ids) {

                    $taxonomy_key = sanitize_key($taxonomy_key);

                    $taxonomy = end(explode('_', $taxonomy_key));

                    if ( !is_array($categories_ids) ) { $categories_ids = [$categories_ids]; }

                    $categories_ids = array_map('intval', $categories_ids);

                    wp_set_object_terms($post_id, $categories_ids, $taxonomy);
                }

            } else {

                if ( ! empty( $post_categories ) ) {

                    $category_ids = [];

                    foreach ( $post_categories as $cat_name ) {

                        $term = term_exists( $cat_name, 'category' );

                        if ( $term === 0 || $term === null ) {

                            $term = wp_insert_term( $cat_name, 'category' );
                        }

                        if ( ! is_wp_error( $term ) ) {
                            
                            if ( is_array( $term ) ) {

                                $category_ids[] = $term['term_id'];
                                
                            } else {

                                $category_ids[] = (int) $term;
                            }
                        }
                    }

                    wp_set_post_categories( $post_id, $category_ids );
                }

            }

            if( !empty($post_tags) ) { wp_set_post_tags($post_id, implode(',', $post_tags)); }

            if( !empty($post_meta_title) && !empty($post_meta_description) ) { 

                update_post_meta($post_id, '_meta_title', $post_meta_title);
                update_post_meta($post_id, '_meta_description', $post_meta_description);

                if (defined('WPSEO_VERSION')) {

                    update_post_meta( $post_id, '_yoast_wpseo_focuskw', $post_meta_title );
                    update_post_meta($post_id, '_yoast_wpseo_title', $post_meta_title);
                    update_post_meta($post_id, '_yoast_wpseo_metadesc', $post_meta_description);

                }
            }

            if( !empty($article_image_id) ) {

                set_post_thumbnail($post_id, $article_image_id);

            } else {

                if ( !empty($post_image) && !empty($post_alt) ) {

                    $image_data = wp_remote_get($post_image, ['timeout' => 30]);

                    if (!is_wp_error($image_data)) {

                        $contents = wp_remote_retrieve_body($image_data);
                        $file_name = basename(parse_url($post_image, PHP_URL_PATH));
                        $upload = wp_upload_bits($file_name, null, $contents);

                        if (!$upload['error']) {
                            
                            $file_array = [
                                'name'     => $file_name,
                                'tmp_name' => $upload['file'],
                            ];

                            $id = media_handle_sideload($file_array, $post_id, null, [
                                'post_title'   => $post_alt,
                                'post_excerpt' => $post_alt,
                                'post_content' => $post_alt,
                            ]);

                            if (!is_wp_error($id)) {

                                set_post_thumbnail($post_id, $id);

                                update_post_meta($id, '_wp_attachment_image_alt', $post_alt);

                            } else {

                                @unlink($upload['file']);
                            }
                        }
                    }
                    
                }
            }

            $smart_plugin_articles_generated_count = (int) get_user_meta(1, 'smart_plugin_articles_generated', true);

            $smart_plugin_articles_generated_count += 1;

            update_user_meta(1, 'smart_plugin_articles_generated', $smart_plugin_articles_generated_count);

            if( !empty($schedule_list) ) {

                $schedules = get_user_meta(1, 'schedule_lists', true);

                $schedules[$schedule_list]['done'] = true;

                update_user_meta(1, 'schedule_lists', $schedules);

            }

            wp_send_json_success('<p>Article generated and published successfully. <a href="' . get_permalink($post_id) . '" target="_blank">View Article</a></p>');
            
        } else {

            $message = !empty($body['message']) ? $body['message'] : 'Unknown error';

            wp_send_json_error('<p>' . esc_html($message) . '</p>');
        }

    } catch (RequestException $e) {
        $error = $e->hasResponse()
            ? $e->getResponse()->getBody()->getContents()
            : $e->getMessage();

        $decoded = json_decode($error, true);

        $message = isset($decoded['message']) ? $decoded['message'] : $error;

        wp_send_json_error('<p>Request failed: ' . esc_html($message) . '</p>');
    }
}
